<?php
/**
 * Update Subscriber Custom Field
 *
 * @package     AutomatorWP\Integrations\AWeber\Actions\Update_Subscriber_Custom_Field
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined('ABSPATH') ) exit;


class AutomatorWP_AWeber_Update_Subscriber_Custom_Field extends AutomatorWP_Integration_Action {
    
    public $integration = 'aweber';
    public $action = 'aweber_update_subscriber_custom_field';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration' => $this->integration,
            'label' => __( 'Update subscriber custom field', 'automatorwp-pro' ),
            'select_option' => __( 'Update <strong>subscriber</strong> custom field', 'automatorwp-pro' ),
            /* translators: %1$s: Subscriber. %2$s: Custom field name. %3$s: Custom field value.*/
            'edit_label'        => sprintf( __( 'Update %1$s %2$s field with the value %3$s', 'automatorwp-pro' ), '{subscriber}', '{custom_field}', '{custom_field_value}' ),
            /* translators: %1$s: Custom field name. %2$s: Custom field value. */
            'log_label'         => sprintf( __( 'Update subscriber %1$s field with the value %2$s', 'automatorwp-pro' ), '{custom_field}', '{custom_field_value}' ),
            'options'           => array(
                'custom_field' => array(
                    'from' => 'custom_field',
                    'default' => __( 'custom', 'automatorwp-pro' ),
                    'fields' => array(
                        'account' => automatorwp_utilities_ajax_selector_field( array(
                            'name'              => __( 'Account:', 'automatorwp-pro' ),
                            'option_none'       => false,
                            'action_cb'         => 'automatorwp_aweber_get_accounts',
                            'options_cb'        => 'automatorwp_aweber_options_cb_account',
                            'placeholder'       => 'Select an account',
                            'default'           => ''
                        ) ),
                        'lists' => automatorwp_utilities_ajax_selector_field( array(
                            'name'              => __( 'List:', 'automatorwp-pro' ),
                            'option_none'       => false,
                            'action_cb'         => 'automatorwp_aweber_get_lists',
                            'options_cb'        => 'automatorwp_aweber_options_cb_list',
                            'placeholder'       => 'Select a list',
                            'default'           => ''
                        ) ),
                        'custom_field' => automatorwp_utilities_ajax_selector_field( array(
                            'name'              => __( 'Custom field:', 'automatorwp-pro' ),
                            'option_none'       => false,
                            'action_cb'         => 'automatorwp_aweber_get_custom_fields',
                            'options_cb'        => 'automatorwp_aweber_options_cb_custom_field',
                            'placeholder'       => 'Select a custom field',
                            'default'           => ''
                        ) ),      
                    ),
                ),
                'custom_field_value' => array(
                    'from' => 'custom_field_value',
                    'default' => __( 'value', 'automatorwp-pro' ),
                    'fields' => array(
                        'custom_field_value' => array(
                            'name' => __( 'Value:', 'automatorwp-pro' ),
                            'desc' => __( 'The value to update the custom field.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                     )
                ),
                'subscriber' => array(
                    'from' => 'subscriber',
                    'default' => __( 'subscriber', 'automatorwp-pro' ),
                    'fields' => array(
                        'subscriber' => array(
                            'name' => __( 'Subscriber:', 'automatorwp-pro' ),
                            'desc' => __( 'The subscriber email.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                     )
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation){

        $params = automatorwp_aweber_get_request_parameters();

        // Bail if the authorization has not been setup from settings
        if( $params === false ) {
            $this->result = __( 'AWeber integration not configured in AutomatorWP settings.', 'automatorwp-pro' );
            return;
        }

        // Shorthand
        $user = get_user_by ( 'ID', $user_id );
        $subscriber_email = $action_options['subscriber'];
        $account_id = $action_options['account'];
        $list_id = $action_options['lists'];
        $custom_field_id = $action_options['custom_field'];
        $custom_field_value = $action_options['custom_field_value'];

        // Bail if not email
        if ( empty( $subscriber_email ) ){
            $this->result = __( 'Email subscriber field is empty.', 'automatorwp-pro' );
            return;
        }

        // Get the custom field name
        $custom_field_name = automatorwp_aweber_get_custom_field_name( $account_id, $list_id, $custom_field_id );

        // Get custom fields from subscriber
        $custom_fields_subscriber = automatorwp_aweber_get_subscriber_custom_fields( $account_id, $list_id, $subscriber_email );

        // Update the specific custom field
        if ( array_key_exists ( $custom_field_name, $custom_fields_subscriber ) ) {
            $custom_fields_subscriber[$custom_field_name] = $custom_field_value; 
        } else {
            $this->result = __( 'Custom field not available.', 'automatorwp-pro' );
            return;
        }

        $custom_field = array(
            'custom_fields' => $custom_fields_subscriber,
        );

        $response = automatorwp_aweber_update_custom_field( $custom_field, $account_id, $list_id, $subscriber_email );

        if ( isset( $response['error'] ) ){
            $this->result = sprintf( __( '%s could not be added to the list in AWeber. Error: %s, %s', 'automatorwp-pro' ), $subscriber_email, $response['error']['status'], $response['error']['message'] );
        } else {
            $this->result = sprintf( __( '%s added to list in AWeber', 'automatorwp-pro' ), $subscriber_email );
        }
        
    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }
    
}

new AutomatorWP_AWeber_Update_Subscriber_Custom_Field();